unit XSortLst;
{
  ====================
  Sort String List 1.0 (1999-01-09)
  ====================
  TStrings descendant with general sorting functionality.

  TSortStringList
  ---------------
  Uses Borland version of quick sort (with small modification).
  You can also define QUICK_SORT_UNDU and use Quick Sort version published
  in UNDU.

  public methods

    Sort			sort whole string list
    QuickSort			sort a subrange of string list
    StdCompareFunction          alphabetical sort function (default)

  public properties

    List			direct acces to list read only
    CompareFunction		custom compare function (procedural type)

  Freeware.

  Copyright  Roman Stedronsky 1998, Roman.Stedronsky@seznam.cz

  All rights reserved. You may use this software in an application
  without fee or royalty, provided this copyright notice remains intact.
}

{ $DEFINE QUICK_SORT_UNDU}

interface

uses
  Classes, Messages;

type
  TSortStringListCompare = function(const V1, V2: integer): integer of object;

  TSortStringList = class(TStrings)
  private
    FList: PStringItemList;
    FCount: Integer;
    FCapacity: Integer;
    FSorted: Boolean;
    FDuplicates: TDuplicates;
    FOnChange: TNotifyEvent;
    FOnChanging: TNotifyEvent;
    procedure ExchangeItems(Index1, Index2: Integer);
    procedure Grow;
    procedure InsertItem(Index: Integer; const S: string);
    procedure SetSorted(Value: Boolean);
  protected
    procedure Changed; virtual;
    procedure Changing; virtual;
    function Get(Index: Integer): string; override;
    function GetCapacity: Integer; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure Put(Index: Integer; const S: string); override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetCapacity(NewCapacity: Integer); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    constructor Create; virtual;
    destructor Destroy; override;
    function Add(const S: string): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Exchange(Index1, Index2: Integer); override;
    function Find(const S: string; var Index: Integer): Boolean; virtual;
    function IndexOf(const S: string): Integer; override;
    procedure Insert(Index: Integer; const S: string); override;
    procedure QuickSort(L, R: Integer);
    procedure Sort; virtual;
    property Duplicates: TDuplicates read FDuplicates write FDuplicates;
    property Sorted: Boolean read FSorted write SetSorted;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnChanging: TNotifyEvent read FOnChanging write FOnChanging;
    property List: PStringItemList read FList; //added
  private
    FCompareFunction: TSortStringListCompare;
  public
    property CompareFunction: TSortStringListCompare read FCompareFunction write FCompareFunction;
    function StdCompareFunction(const V1, V2: integer): integer;
  end;

function CompareInt(i1, i2: integer): integer;

implementation

uses
  Consts, SysUtils, Windows;

function CompareInt(i1, i2: integer): integer;
asm
 sub EAX,EDX
 jz @@konec
 jb @@less
 mov EAX,1
 jmp @@konec
@@less:
 mov EAX,-1
@@konec:
end;

constructor TSortStringList.Create;
begin
  inherited Create;
  FCompareFunction := StdCompareFunction;
end;

destructor TSortStringList.Destroy;
begin
  FOnChange := nil;
  FOnChanging := nil;
  inherited Destroy;
  if FCount <> 0 then Finalize(FList^[0], FCount);
  FCount := 0;
  SetCapacity(0);
end;

function TSortStringList.Add(const S: string): Integer;
begin
  if not Sorted then
    Result := FCount
  else if Find(S, Result) then
    case Duplicates of
      dupIgnore: Exit;
      dupError: Error(SDuplicateString, 0);
    end;
  InsertItem(Result, S);
end;

procedure TSortStringList.Changed;
begin
  if {(FUpdateCount = 0) and}  Assigned(FOnChange) then FOnChange(Self);
end;

procedure TSortStringList.Changing;
begin
  if {(FUpdateCount = 0) and}  Assigned(FOnChanging) then FOnChanging(Self);
end;

procedure TSortStringList.Clear;
begin
  if FCount <> 0 then
  begin
    Changing;
    Finalize(FList^[0], FCount);
    FCount := 0;
    SetCapacity(0);
    Changed;
  end;
end;

procedure TSortStringList.Delete(Index: Integer);
begin
  if (Index < 0) or (Index >= FCount) then Error(SListIndexError, Index);
  Changing;
  Finalize(FList^[Index]);
  Dec(FCount);
  if Index < FCount then
    System.Move(FList^[Index + 1], FList^[Index],
      (FCount - Index) * SizeOf(TStringItem));
  Changed;
end;

procedure TSortStringList.Exchange(Index1, Index2: Integer);
begin
  if (Index1 < 0) or (Index1 >= FCount) then Error(SListIndexError, Index1);
  if (Index2 < 0) or (Index2 >= FCount) then Error(SListIndexError, Index2);
  Changing;
  ExchangeItems(Index1, Index2);
  Changed;
end;

procedure TSortStringList.ExchangeItems(Index1, Index2: Integer);
var
  Temp: Integer;
  Item1, Item2: PStringItem;
begin
  Item1 := @FList^[Index1];
  Item2 := @FList^[Index2];
  Temp := Integer(Item1^.FString);
  Integer(Item1^.FString) := Integer(Item2^.FString);
  Integer(Item2^.FString) := Temp;
  Temp := Integer(Item1^.FObject);
  Integer(Item1^.FObject) := Integer(Item2^.FObject);
  Integer(Item2^.FObject) := Temp;
end;

function TSortStringList.Find(const S: string; var Index: Integer): Boolean;
var
  L, H, I, C: Integer;
begin
  Result := False;
  L := 0;
  H := FCount - 1;
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := AnsiCompareText(FList^[I].FString, S);
//    C := CompareFunction(I, S);
    if C < 0 then
      L := I + 1
    else
    begin
      H := I - 1;
      if C = 0 then
      begin
        Result := True;
        if Duplicates <> dupAccept then L := I;
      end;
    end;
  end;
  Index := L;
end;

function TSortStringList.Get(Index: Integer): string;
begin
  if (Index < 0) or (Index >= FCount) then Error(SListIndexError, Index);
  Result := FList^[Index].FString;
end;

function TSortStringList.GetCapacity: Integer;
begin
  Result := FCapacity;
end;

function TSortStringList.GetCount: Integer;
begin
  Result := FCount;
end;

function TSortStringList.GetObject(Index: Integer): TObject;
begin
  if (Index < 0) or (Index >= FCount) then Error(SListIndexError, Index);
  Result := FList^[Index].FObject;
end;

procedure TSortStringList.Grow;
var
  Delta: Integer;
begin
  if FCapacity > 64 then
    Delta := FCapacity div 4
  else if FCapacity > 8 then
    Delta := 16
  else
    Delta := 4;
  SetCapacity(FCapacity + Delta);
end;

function TSortStringList.IndexOf(const S: string): Integer;
begin
  if not Sorted then
    Result := inherited IndexOf(S)
  else if not Find(S, Result) then
    Result := -1;
end;

procedure TSortStringList.Insert(Index: Integer; const S: string);
begin
  if Sorted then Error(SSortedListError, 0);
  if (Index < 0) or (Index > FCount) then Error(SListIndexError, Index);
  InsertItem(Index, S);
end;

procedure TSortStringList.InsertItem(Index: Integer; const S: string);
begin
  Changing;
  if FCount = FCapacity then Grow;
  if Index < FCount then
    System.Move(FList^[Index], FList^[Index + 1],
      (FCount - Index) * SizeOf(TStringItem));
  with FList^[Index] do
  begin
    Pointer(FString) := nil;
    FObject := nil;
    FString := S;
  end;
  Inc(FCount);
  Changed;
end;

procedure TSortStringList.Put(Index: Integer; const S: string);
begin
  if Sorted then Error(SSortedListError, 0);
  if (Index < 0) or (Index >= FCount) then Error(SListIndexError, Index);
  Changing;
  FList^[Index].FString := S;
  Changed;
end;

procedure TSortStringList.PutObject(Index: Integer; AObject: TObject);
begin
  if (Index < 0) or (Index >= FCount) then Error(SListIndexError, Index);
  Changing;
  FList^[Index].FObject := AObject;
  Changed;
end;

{$IFNDEF QUICK_SORT_UNDU}

procedure TSortStringList.QuickSort(L, R: Integer);
var
  I, J, M: Integer;
begin
  I := L;
  J := R;
  M := (L + R) shr 1;
  repeat
    while CompareFunction(I, M) < 0 do Inc(I);
    while CompareFunction(J, M) > 0 do Dec(J);
    if I <= J then
    begin
      ExchangeItems(I, J);
      if I = M then M := J else if J = M then M := I; // keeeping VALUE is absolutely necessary
      Inc(I);
      Dec(J);
    end;
  until I > J;
  if J > L then QuickSort(L, J);
  if I < R then QuickSort(I, R);
end;

{$ELSE}

procedure TSortStringList.QuickSort(L, R: integer);
var
  Index, M, P: integer;
begin
  if L < R then
  begin
    P := L;
    M := L;
    for Index := L + 1 to R do
    begin
      if CompareFunction(Index, P) < 0 then
      begin
        Inc(M);
        ExchangeItems(Index, M);
      end;
    end;
    ExchangeItems(L, M);
    QuickSort(L, M - 1);
    QuickSort(M + 1, R);
  end;
end;

{$ENDIF}

procedure TSortStringList.SetCapacity(NewCapacity: Integer);
begin
  ReallocMem(FList, NewCapacity * SizeOf(TStringItem));
  FCapacity := NewCapacity;
end;

procedure TSortStringList.SetSorted(Value: Boolean);
begin
  if FSorted <> Value then
  begin
    if Value then Sort;
    FSorted := Value;
  end;
end;

procedure TSortStringList.SetUpdateState(Updating: Boolean);
begin
  if Updating then
    Changing
  else
    Changed;
end;

procedure TSortStringList.Sort;
begin
  if not Sorted and (FCount > 1) then
  begin
    Changing;
    QuickSort(0, FCount - 1);
    Changed;
  end;
end;

function TSortStringList.StdCompareFunction(const V1, V2: integer): integer;
begin
  Result := AnsiCompareText(FList^[V1].FString, FList^[V2].FString);
end;

end.




